<?php

namespace Database\Seeders;

use App\Enums\BatchStatus;
use App\Models\Category;
use App\Models\Product;
use App\Models\ProductBatch;
use App\Models\Supplier;
use App\Models\Unit;
use Illuminate\Database\Seeder;

class ProductSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        // Get units
        $units = [
            'tablet' => Unit::where('name', 'Tablet')->first(),
            'kapsul' => Unit::where('name', 'Kapsul')->first(),
            'botol' => Unit::where('name', 'Botol')->first(),
            'tube' => Unit::where('name', 'Tube')->first(),
            'sachet' => Unit::where('name', 'Sachet')->first(),
            'ampul' => Unit::where('name', 'Ampul')->first(),
            'strip' => Unit::where('name', 'Strip')->first(),
            'box' => Unit::where('name', 'Box')->first(),
            'pcs' => Unit::where('name', 'Piece')->first(),
            'inhaler' => Unit::where('name', 'Inhaler')->first(),
        ];

        // Get categories
        $categories = [
            'obat_demam' => Category::where('name', 'Obat Demam')->first(),
            'obat_flu' => Category::where('name', 'Obat Flu & Pilek')->first(),
            'antibiotik' => Category::where('name', 'Antibiotik')->first(),
            'vitamin' => Category::where('name', 'Vitamin & Suplemen')->first(),
            'obat_batuk' => Category::where('name', 'Obat Batuk')->first(),
            'obat_maag' => Category::where('name', 'Obat Maag')->first(),
            'obat_alergi' => Category::where('name', 'Obat Alergi')->first(),
            'obat_diare' => Category::where('name', 'Obat Diare')->first(),
            'obat_mata' => Category::where('name', 'Obat Mata')->first(),
            'antihipertensi' => Category::where('name', 'Antihipertensi')->first(),
            'antidiabetes' => Category::where('name', 'Antidiabetes')->first(),
            'obat_jantung' => Category::where('name', 'Obat Jantung')->first(),
            'alat_tes' => Category::where('name', 'Alat Tes')->first(),
            'masker' => Category::where('name', 'Masker & APD')->first(),
            'perban' => Category::where('name', 'Perban & Plester')->first(),
            'perawatan_kulit' => Category::where('name', 'Perawatan Kulit')->first(),
            'perawatan_rambut' => Category::where('name', 'Perawatan Rambut')->first(),
            'jamu' => Category::where('name', 'Jamu')->first(),
            'susu' => Category::where('name', 'Susu & Nutrisi')->first(),
            'obat_sakit_kepala' => Category::where('name', 'Obat Sakit Kepala')->first(),
            'obat_luka' => Category::where('name', 'Obat Luka')->first(),
        ];

        $suppliers = Supplier::all();

        $products = [
            // === OBAT BEBAS (10 produk) ===
            [
                'code' => 'OB001',
                'barcode' => '8991234567001',
                'name' => 'Paracetamol 500mg',
                'generic_name' => 'Paracetamol',
                'category' => 'obat_demam',
                'unit' => 'tablet',
                'purchase_price' => 150,
                'selling_price' => 500,
                'min_stock' => 100,
                'rack_location' => 'A-01-01',
            ],
            [
                'code' => 'OB002',
                'barcode' => '8991234567002',
                'name' => 'Bodrex Tablet',
                'generic_name' => 'Paracetamol, Caffeine',
                'category' => 'obat_sakit_kepala',
                'unit' => 'tablet',
                'purchase_price' => 200,
                'selling_price' => 600,
                'min_stock' => 100,
                'rack_location' => 'A-01-02',
            ],
            [
                'code' => 'OB003',
                'barcode' => '8991234567003',
                'name' => 'Promag Tablet',
                'generic_name' => 'Antasida, Simethicone',
                'category' => 'obat_maag',
                'unit' => 'tablet',
                'purchase_price' => 200,
                'selling_price' => 600,
                'min_stock' => 100,
                'rack_location' => 'A-01-03',
            ],
            [
                'code' => 'OB004',
                'barcode' => '8991234567004',
                'name' => 'Mylanta Sirup 150ml',
                'generic_name' => 'Aluminium Hydroxide, Magnesium Hydroxide',
                'category' => 'obat_maag',
                'unit' => 'botol',
                'purchase_price' => 18000,
                'selling_price' => 32000,
                'min_stock' => 30,
                'rack_location' => 'A-01-04',
            ],
            [
                'code' => 'OB005',
                'barcode' => '8991234567005',
                'name' => 'Antangin JRG Cair',
                'generic_name' => 'Jahe, Royal Jelly, Ginseng',
                'category' => 'obat_demam',
                'unit' => 'sachet',
                'purchase_price' => 1500,
                'selling_price' => 2500,
                'min_stock' => 50,
                'rack_location' => 'A-02-01',
            ],
            [
                'code' => 'OB006',
                'barcode' => '8991234567006',
                'name' => 'OBH Combi Batuk Berdahak',
                'generic_name' => 'Succus Liquiritiae, Ammonium Chloride',
                'category' => 'obat_batuk',
                'unit' => 'botol',
                'purchase_price' => 12000,
                'selling_price' => 22000,
                'min_stock' => 25,
                'rack_location' => 'A-02-02',
            ],
            [
                'code' => 'OB007',
                'barcode' => '8991234567007',
                'name' => 'Komix Herbal Tube',
                'generic_name' => 'Herbal Extract',
                'category' => 'obat_batuk',
                'unit' => 'sachet',
                'purchase_price' => 1800,
                'selling_price' => 3000,
                'min_stock' => 50,
                'rack_location' => 'A-02-03',
            ],
            [
                'code' => 'OB008',
                'barcode' => '8991234567008',
                'name' => 'Paracetamol Sirup 60ml',
                'generic_name' => 'Paracetamol',
                'category' => 'obat_demam',
                'unit' => 'botol',
                'purchase_price' => 8000,
                'selling_price' => 15000,
                'min_stock' => 30,
                'rack_location' => 'A-03-01',
            ],
            [
                'code' => 'OB009',
                'barcode' => '8991234567009',
                'name' => 'Betadine Antiseptik 30ml',
                'generic_name' => 'Povidone Iodine',
                'category' => 'obat_luka',
                'unit' => 'botol',
                'purchase_price' => 22000,
                'selling_price' => 38000,
                'min_stock' => 20,
                'rack_location' => 'A-03-02',
            ],
            [
                'code' => 'OB010',
                'barcode' => '8991234567010',
                'name' => 'Minyak Kayu Putih Cap Lang 60ml',
                'generic_name' => 'Cajuput Oil',
                'category' => 'obat_demam',
                'unit' => 'botol',
                'purchase_price' => 18000,
                'selling_price' => 28000,
                'min_stock' => 25,
                'rack_location' => 'A-03-03',
            ],

            // === OBAT BEBAS TERBATAS (10 produk) ===
            [
                'code' => 'OBT001',
                'barcode' => '8991234567011',
                'name' => 'Decolgen Tablet',
                'generic_name' => 'Paracetamol, Phenylpropanolamine, CTM',
                'category' => 'obat_flu',
                'unit' => 'tablet',
                'purchase_price' => 300,
                'selling_price' => 800,
                'min_stock' => 80,
                'rack_location' => 'B-01-01',
            ],
            [
                'code' => 'OBT002',
                'barcode' => '8991234567012',
                'name' => 'Neozep Forte',
                'generic_name' => 'Paracetamol, Phenylephrine, CTM',
                'category' => 'obat_flu',
                'unit' => 'tablet',
                'purchase_price' => 350,
                'selling_price' => 900,
                'min_stock' => 80,
                'rack_location' => 'B-01-02',
            ],
            [
                'code' => 'OBT003',
                'barcode' => '8991234567013',
                'name' => 'CTM 4mg',
                'generic_name' => 'Chlorpheniramine Maleate',
                'category' => 'obat_alergi',
                'unit' => 'tablet',
                'purchase_price' => 100,
                'selling_price' => 300,
                'min_stock' => 100,
                'rack_location' => 'B-01-03',
            ],
            [
                'code' => 'OBT004',
                'barcode' => '8991234567014',
                'name' => 'Loratadine 10mg',
                'generic_name' => 'Loratadine',
                'category' => 'obat_alergi',
                'unit' => 'tablet',
                'purchase_price' => 400,
                'selling_price' => 1000,
                'min_stock' => 60,
                'rack_location' => 'B-02-01',
            ],
            [
                'code' => 'OBT005',
                'barcode' => '8991234567015',
                'name' => 'New Diatabs',
                'generic_name' => 'Attapulgite',
                'category' => 'obat_diare',
                'unit' => 'tablet',
                'purchase_price' => 250,
                'selling_price' => 700,
                'min_stock' => 80,
                'rack_location' => 'B-02-02',
            ],
            [
                'code' => 'OBT006',
                'barcode' => '8991234567016',
                'name' => 'Entrostop',
                'generic_name' => 'Attapulgite, Pectin',
                'category' => 'obat_diare',
                'unit' => 'tablet',
                'purchase_price' => 300,
                'selling_price' => 800,
                'min_stock' => 80,
                'rack_location' => 'B-02-03',
            ],
            [
                'code' => 'OBT007',
                'barcode' => '8991234567017',
                'name' => 'Insto Regular',
                'generic_name' => 'Tetrahydrozoline HCl',
                'category' => 'obat_mata',
                'unit' => 'botol',
                'purchase_price' => 12000,
                'selling_price' => 22000,
                'min_stock' => 30,
                'rack_location' => 'B-03-01',
            ],
            [
                'code' => 'OBT008',
                'barcode' => '8991234567018',
                'name' => 'Rohto Cool',
                'generic_name' => 'Zinc Sulfate, Tetrahydrozoline',
                'category' => 'obat_mata',
                'unit' => 'botol',
                'purchase_price' => 15000,
                'selling_price' => 28000,
                'min_stock' => 30,
                'rack_location' => 'B-03-02',
            ],
            [
                'code' => 'OBT009',
                'barcode' => '8991234567019',
                'name' => 'Voltaren Emulgel 20g',
                'generic_name' => 'Diclofenac Diethylamine',
                'category' => 'obat_luka',
                'unit' => 'tube',
                'purchase_price' => 45000,
                'selling_price' => 75000,
                'min_stock' => 15,
                'rack_location' => 'B-03-03',
            ],
            [
                'code' => 'OBT010',
                'barcode' => '8991234567020',
                'name' => 'Counterpain 30g',
                'generic_name' => 'Methyl Salicylate, Menthol',
                'category' => 'obat_luka',
                'unit' => 'tube',
                'purchase_price' => 28000,
                'selling_price' => 48000,
                'min_stock' => 20,
                'rack_location' => 'B-04-01',
            ],

            // === OBAT KERAS (10 produk) ===
            [
                'code' => 'OK001',
                'barcode' => '8991234567021',
                'name' => 'Amoxicillin 500mg',
                'generic_name' => 'Amoxicillin',
                'category' => 'antibiotik',
                'unit' => 'kapsul',
                'purchase_price' => 800,
                'selling_price' => 1500,
                'min_stock' => 50,
                'rack_location' => 'C-01-01',
                'requires_prescription' => true,
            ],
            [
                'code' => 'OK002',
                'barcode' => '8991234567022',
                'name' => 'Cefixime 100mg',
                'generic_name' => 'Cefixime',
                'category' => 'antibiotik',
                'unit' => 'kapsul',
                'purchase_price' => 2500,
                'selling_price' => 4500,
                'min_stock' => 30,
                'rack_location' => 'C-01-02',
                'requires_prescription' => true,
            ],
            [
                'code' => 'OK003',
                'barcode' => '8991234567023',
                'name' => 'Ciprofloxacin 500mg',
                'generic_name' => 'Ciprofloxacin',
                'category' => 'antibiotik',
                'unit' => 'tablet',
                'purchase_price' => 1200,
                'selling_price' => 2500,
                'min_stock' => 40,
                'rack_location' => 'C-01-03',
                'requires_prescription' => true,
            ],
            [
                'code' => 'OK004',
                'barcode' => '8991234567024',
                'name' => 'Amlodipine 5mg',
                'generic_name' => 'Amlodipine Besylate',
                'category' => 'antihipertensi',
                'unit' => 'tablet',
                'purchase_price' => 500,
                'selling_price' => 1200,
                'min_stock' => 50,
                'rack_location' => 'C-02-01',
                'requires_prescription' => true,
            ],
            [
                'code' => 'OK005',
                'barcode' => '8991234567025',
                'name' => 'Captopril 25mg',
                'generic_name' => 'Captopril',
                'category' => 'antihipertensi',
                'unit' => 'tablet',
                'purchase_price' => 300,
                'selling_price' => 800,
                'min_stock' => 60,
                'rack_location' => 'C-02-02',
                'requires_prescription' => true,
            ],
            [
                'code' => 'OK006',
                'barcode' => '8991234567026',
                'name' => 'Metformin 500mg',
                'generic_name' => 'Metformin HCl',
                'category' => 'antidiabetes',
                'unit' => 'tablet',
                'purchase_price' => 400,
                'selling_price' => 1000,
                'min_stock' => 80,
                'rack_location' => 'C-02-03',
                'requires_prescription' => true,
            ],
            [
                'code' => 'OK007',
                'barcode' => '8991234567027',
                'name' => 'Glimepiride 2mg',
                'generic_name' => 'Glimepiride',
                'category' => 'antidiabetes',
                'unit' => 'tablet',
                'purchase_price' => 800,
                'selling_price' => 1800,
                'min_stock' => 40,
                'rack_location' => 'C-03-01',
                'requires_prescription' => true,
            ],
            [
                'code' => 'OK008',
                'barcode' => '8991234567028',
                'name' => 'Omeprazole 20mg',
                'generic_name' => 'Omeprazole',
                'category' => 'obat_maag',
                'unit' => 'kapsul',
                'purchase_price' => 1200,
                'selling_price' => 2500,
                'min_stock' => 40,
                'rack_location' => 'C-03-02',
                'requires_prescription' => true,
            ],
            [
                'code' => 'OK009',
                'barcode' => '8991234567029',
                'name' => 'Lansoprazole 30mg',
                'generic_name' => 'Lansoprazole',
                'category' => 'obat_maag',
                'unit' => 'kapsul',
                'purchase_price' => 1500,
                'selling_price' => 3000,
                'min_stock' => 30,
                'rack_location' => 'C-03-03',
                'requires_prescription' => true,
            ],
            [
                'code' => 'OK010',
                'barcode' => '8991234567030',
                'name' => 'Simvastatin 10mg',
                'generic_name' => 'Simvastatin',
                'category' => 'obat_jantung',
                'unit' => 'tablet',
                'purchase_price' => 600,
                'selling_price' => 1500,
                'min_stock' => 50,
                'rack_location' => 'C-04-01',
                'requires_prescription' => true,
            ],

            // === VITAMIN & SUPLEMEN (10 produk) ===
            [
                'code' => 'VS001',
                'barcode' => '8991234567031',
                'name' => 'Vitamin C 500mg IPI',
                'generic_name' => 'Ascorbic Acid',
                'category' => 'vitamin',
                'unit' => 'tablet',
                'purchase_price' => 400,
                'selling_price' => 1000,
                'min_stock' => 100,
                'rack_location' => 'D-01-01',
            ],
            [
                'code' => 'VS002',
                'barcode' => '8991234567032',
                'name' => 'Enervon-C',
                'generic_name' => 'Vitamin B Complex, Vitamin C',
                'category' => 'vitamin',
                'unit' => 'tablet',
                'purchase_price' => 600,
                'selling_price' => 1200,
                'min_stock' => 80,
                'rack_location' => 'D-01-02',
            ],
            [
                'code' => 'VS003',
                'barcode' => '8991234567033',
                'name' => 'Blackmores Fish Oil 1000mg',
                'generic_name' => 'Omega-3 Fish Oil',
                'category' => 'vitamin',
                'unit' => 'kapsul',
                'purchase_price' => 3500,
                'selling_price' => 5500,
                'min_stock' => 20,
                'rack_location' => 'D-01-03',
            ],
            [
                'code' => 'VS004',
                'barcode' => '8991234567034',
                'name' => 'Becom-C',
                'generic_name' => 'Vitamin B Complex, Vitamin C',
                'category' => 'vitamin',
                'unit' => 'tablet',
                'purchase_price' => 500,
                'selling_price' => 1100,
                'min_stock' => 80,
                'rack_location' => 'D-02-01',
            ],
            [
                'code' => 'VS005',
                'barcode' => '8991234567035',
                'name' => 'Neurobion Forte',
                'generic_name' => 'Vitamin B1, B6, B12',
                'category' => 'vitamin',
                'unit' => 'tablet',
                'purchase_price' => 1800,
                'selling_price' => 3500,
                'min_stock' => 40,
                'rack_location' => 'D-02-02',
            ],
            [
                'code' => 'VS006',
                'barcode' => '8991234567036',
                'name' => 'Calcium Lactate 500mg',
                'generic_name' => 'Calcium Lactate',
                'category' => 'vitamin',
                'unit' => 'tablet',
                'purchase_price' => 300,
                'selling_price' => 800,
                'min_stock' => 80,
                'rack_location' => 'D-02-03',
            ],
            [
                'code' => 'VS007',
                'barcode' => '8991234567037',
                'name' => 'Sangobion',
                'generic_name' => 'Iron, Vitamin B12, Folic Acid',
                'category' => 'vitamin',
                'unit' => 'kapsul',
                'purchase_price' => 1200,
                'selling_price' => 2500,
                'min_stock' => 50,
                'rack_location' => 'D-03-01',
            ],
            [
                'code' => 'VS008',
                'barcode' => '8991234567038',
                'name' => 'Hemaviton Stamina Plus',
                'generic_name' => 'Multivitamin, Ginseng',
                'category' => 'vitamin',
                'unit' => 'kapsul',
                'purchase_price' => 1500,
                'selling_price' => 3000,
                'min_stock' => 40,
                'rack_location' => 'D-03-02',
            ],
            [
                'code' => 'VS009',
                'barcode' => '8991234567039',
                'name' => 'CDR Calcium D Redoxon',
                'generic_name' => 'Calcium, Vitamin D, Vitamin C',
                'category' => 'vitamin',
                'unit' => 'tablet',
                'purchase_price' => 2200,
                'selling_price' => 4000,
                'min_stock' => 30,
                'rack_location' => 'D-03-03',
            ],
            [
                'code' => 'VS010',
                'barcode' => '8991234567040',
                'name' => 'Vitamin E 400 IU',
                'generic_name' => 'Alpha-Tocopherol',
                'category' => 'vitamin',
                'unit' => 'kapsul',
                'purchase_price' => 800,
                'selling_price' => 1800,
                'min_stock' => 50,
                'rack_location' => 'D-04-01',
            ],

            // === ALAT KESEHATAN (5 produk) ===
            [
                'code' => 'AK001',
                'barcode' => '8991234567041',
                'name' => 'Test Strip Gula Darah',
                'generic_name' => 'Blood Glucose Test Strip',
                'category' => 'alat_tes',
                'unit' => 'box',
                'purchase_price' => 85000,
                'selling_price' => 125000,
                'min_stock' => 10,
                'rack_location' => 'E-01-01',
            ],
            [
                'code' => 'AK002',
                'barcode' => '8991234567042',
                'name' => 'Masker Medis 3 Ply',
                'generic_name' => 'Surgical Mask 3 Ply',
                'category' => 'masker',
                'unit' => 'box',
                'purchase_price' => 25000,
                'selling_price' => 45000,
                'min_stock' => 20,
                'rack_location' => 'E-01-02',
            ],
            [
                'code' => 'AK003',
                'barcode' => '8991234567043',
                'name' => 'Hand Sanitizer 500ml',
                'generic_name' => 'Ethanol, Glycerin',
                'category' => 'masker',
                'unit' => 'botol',
                'purchase_price' => 35000,
                'selling_price' => 55000,
                'min_stock' => 15,
                'rack_location' => 'E-01-03',
            ],
            [
                'code' => 'AK004',
                'barcode' => '8991234567044',
                'name' => 'Perban Kasa Steril 10cm',
                'generic_name' => 'Sterile Gauze',
                'category' => 'perban',
                'unit' => 'pcs',
                'purchase_price' => 8000,
                'selling_price' => 15000,
                'min_stock' => 30,
                'rack_location' => 'E-02-01',
            ],
            [
                'code' => 'AK005',
                'barcode' => '8991234567045',
                'name' => 'Plester Luka Hansaplast',
                'generic_name' => 'Adhesive Bandage',
                'category' => 'perban',
                'unit' => 'box',
                'purchase_price' => 18000,
                'selling_price' => 32000,
                'min_stock' => 20,
                'rack_location' => 'E-02-02',
            ],

            // === KOSMETIK & PERAWATAN (5 produk) ===
            [
                'code' => 'KS001',
                'barcode' => '8991234567046',
                'name' => 'Vaseline Petroleum Jelly 50g',
                'generic_name' => 'Petroleum Jelly',
                'category' => 'perawatan_kulit',
                'unit' => 'pcs',
                'purchase_price' => 15000,
                'selling_price' => 28000,
                'min_stock' => 20,
                'rack_location' => 'F-01-01',
            ],
            [
                'code' => 'KS002',
                'barcode' => '8991234567047',
                'name' => 'Caladine Lotion 60ml',
                'generic_name' => 'Calamine, Diphenhydramine',
                'category' => 'perawatan_kulit',
                'unit' => 'botol',
                'purchase_price' => 18000,
                'selling_price' => 32000,
                'min_stock' => 20,
                'rack_location' => 'F-01-02',
            ],
            [
                'code' => 'KS003',
                'barcode' => '8991234567048',
                'name' => 'Selsun Blue Shampoo 120ml',
                'generic_name' => 'Selenium Sulfide',
                'category' => 'perawatan_rambut',
                'unit' => 'botol',
                'purchase_price' => 45000,
                'selling_price' => 72000,
                'min_stock' => 15,
                'rack_location' => 'F-01-03',
            ],
            [
                'code' => 'KS004',
                'barcode' => '8991234567049',
                'name' => 'Nizoral Shampoo 80ml',
                'generic_name' => 'Ketoconazole',
                'category' => 'perawatan_rambut',
                'unit' => 'botol',
                'purchase_price' => 65000,
                'selling_price' => 98000,
                'min_stock' => 10,
                'rack_location' => 'F-02-01',
            ],
            [
                'code' => 'KS005',
                'barcode' => '8991234567050',
                'name' => 'Salicyl Talk 60g',
                'generic_name' => 'Salicylic Acid, Talc',
                'category' => 'perawatan_kulit',
                'unit' => 'pcs',
                'purchase_price' => 12000,
                'selling_price' => 22000,
                'min_stock' => 25,
                'rack_location' => 'F-02-02',
            ],

            // === OBAT TRADISIONAL & JAMU (5 produk) ===
            [
                'code' => 'JM001',
                'barcode' => '8991234567051',
                'name' => 'Tolak Angin Cair',
                'generic_name' => 'Herbal Extract',
                'category' => 'jamu',
                'unit' => 'sachet',
                'purchase_price' => 2500,
                'selling_price' => 4500,
                'min_stock' => 50,
                'rack_location' => 'G-01-01',
            ],
            [
                'code' => 'JM002',
                'barcode' => '8991234567052',
                'name' => 'Kuku Bima Ener-G',
                'generic_name' => 'Royal Jelly, Ginseng, Honey',
                'category' => 'jamu',
                'unit' => 'sachet',
                'purchase_price' => 1800,
                'selling_price' => 3000,
                'min_stock' => 50,
                'rack_location' => 'G-01-02',
            ],
            [
                'code' => 'JM003',
                'barcode' => '8991234567053',
                'name' => 'Sido Muncul Kunyit Asam',
                'generic_name' => 'Curcuma, Tamarind',
                'category' => 'jamu',
                'unit' => 'sachet',
                'purchase_price' => 1500,
                'selling_price' => 2500,
                'min_stock' => 50,
                'rack_location' => 'G-01-03',
            ],
            [
                'code' => 'JM004',
                'barcode' => '8991234567054',
                'name' => 'Bintang Toedjoe Masuk Angin',
                'generic_name' => 'Herbal Extract',
                'category' => 'jamu',
                'unit' => 'sachet',
                'purchase_price' => 1200,
                'selling_price' => 2000,
                'min_stock' => 60,
                'rack_location' => 'G-02-01',
            ],
            [
                'code' => 'JM005',
                'barcode' => '8991234567055',
                'name' => 'Extra Joss Active',
                'generic_name' => 'Ginseng, Royal Jelly, Vitamin B',
                'category' => 'jamu',
                'unit' => 'sachet',
                'purchase_price' => 1000,
                'selling_price' => 1800,
                'min_stock' => 80,
                'rack_location' => 'G-02-02',
            ],

            // === SUSU & NUTRISI (5 produk) ===
            [
                'code' => 'SN001',
                'barcode' => '8991234567056',
                'name' => 'Ensure Vanilla 400g',
                'generic_name' => 'Complete Nutrition',
                'category' => 'susu',
                'unit' => 'pcs',
                'purchase_price' => 125000,
                'selling_price' => 175000,
                'min_stock' => 10,
                'rack_location' => 'H-01-01',
            ],
            [
                'code' => 'SN002',
                'barcode' => '8991234567057',
                'name' => 'Diabetasol Vanilla 600g',
                'generic_name' => 'Diabetic Nutrition',
                'category' => 'susu',
                'unit' => 'pcs',
                'purchase_price' => 115000,
                'selling_price' => 165000,
                'min_stock' => 10,
                'rack_location' => 'H-01-02',
            ],
            [
                'code' => 'SN003',
                'barcode' => '8991234567058',
                'name' => 'Entrasol Gold 400g',
                'generic_name' => 'Adult Nutrition',
                'category' => 'susu',
                'unit' => 'pcs',
                'purchase_price' => 85000,
                'selling_price' => 125000,
                'min_stock' => 15,
                'rack_location' => 'H-01-03',
            ],
            [
                'code' => 'SN004',
                'barcode' => '8991234567059',
                'name' => 'Appeton Weight Gain 450g',
                'generic_name' => 'Weight Gain Nutrition',
                'category' => 'susu',
                'unit' => 'pcs',
                'purchase_price' => 165000,
                'selling_price' => 235000,
                'min_stock' => 8,
                'rack_location' => 'H-02-01',
            ],
            [
                'code' => 'SN005',
                'barcode' => '8991234567060',
                'name' => 'Nutren Optimum 400g',
                'generic_name' => 'Complete Nutrition',
                'category' => 'susu',
                'unit' => 'pcs',
                'purchase_price' => 145000,
                'selling_price' => 205000,
                'min_stock' => 10,
                'rack_location' => 'H-02-02',
            ],
        ];

        foreach ($products as $index => $productData) {
            $category = $categories[$productData['category']] ?? null;
            $unit = $units[$productData['unit']] ?? null;

            $product = Product::firstOrCreate(
                ['code' => $productData['code']],
                [
                    'code' => $productData['code'],
                    'barcode' => $productData['barcode'],
                    'name' => $productData['name'],
                    'generic_name' => $productData['generic_name'],
                    'category_id' => $category?->id,
                    'base_unit_id' => $unit?->id,
                    'purchase_price' => $productData['purchase_price'],
                    'selling_price' => $productData['selling_price'],
                    'min_stock' => $productData['min_stock'],
                    'max_stock' => $productData['min_stock'] * 10,
                    'rack_location' => $productData['rack_location'],
                    'requires_prescription' => $productData['requires_prescription'] ?? false,
                    'is_active' => true,
                ]
            );

            // Rotate through suppliers
            $supplier = $suppliers->isNotEmpty() ? $suppliers[$index % $suppliers->count()] : null;

            // Create batches for each product
            $this->createBatches($product, $supplier);
        }
    }

    private function createBatches(Product $product, ?Supplier $supplier): void
    {
        $batches = [
            [
                'batch_number' => 'BTH'.$product->code.'001',
                'expired_date' => now()->addMonths(18),
                'stock' => rand(50, 200),
                'status' => BatchStatus::Active,
            ],
            [
                'batch_number' => 'BTH'.$product->code.'002',
                'expired_date' => now()->addMonths(12),
                'stock' => rand(30, 100),
                'status' => BatchStatus::Active,
            ],
            [
                'batch_number' => 'BTH'.$product->code.'003',
                'expired_date' => now()->addMonths(2),
                'stock' => rand(10, 30),
                'status' => BatchStatus::NearExpired,
            ],
        ];

        foreach ($batches as $batchData) {
            ProductBatch::firstOrCreate(
                [
                    'product_id' => $product->id,
                    'batch_number' => $batchData['batch_number'],
                ],
                [
                    'product_id' => $product->id,
                    'batch_number' => $batchData['batch_number'],
                    'expired_date' => $batchData['expired_date'],
                    'purchase_price' => $product->purchase_price,
                    'selling_price' => $product->selling_price,
                    'stock' => $batchData['stock'],
                    'initial_stock' => $batchData['stock'],
                    'supplier_id' => $supplier?->id,
                    'status' => $batchData['status'],
                ]
            );
        }
    }
}
